# Restaurant Menu Platform

A comprehensive digital menu management system for restaurants with multilingual support and customizable themes.

## Recent Updates

### 🎨 Theme System Implementation

Added a complete theme system that allows restaurants to customize their menu appearance with different color schemes.

**Features:**
- 6 pre-defined color themes:
  1. **Default** - Current orange/white theme
  2. **Classic & Elegant** - Ivory/Gold/Burgundy for refined dining
  3. **Casual & Friendly** - Beige/Tomato red/Mustard for casual restaurants
  4. **Modern & Minimal** - Pure white/Black/Wasabi green for clean look
  5. **Bold & Vibrant** - Warm sand/Chili red/Lime green for energetic vibe
  6. **Dark Mode** - Charcoal/White/Electric blue for modern style

**Implementation Details:**
- New `themes` table stores color schemes with properties:
  - `background_color`, `text_color`, `accent_color`, `cta_color`, `font_style`
- `restaurants` table has `selected_theme` foreign key to themes
- Theme selector in add/edit restaurant forms (accessible by both super_admin and restaurant_admin)
- Dynamic theme loading and application on live menu pages
- CSS custom properties for theme colors
- Helper functions for color manipulation (darken/lighten)

### 🌍 Promotion Items Localization Fix

Fixed the promotion items to properly display localized names (Macedonian/English).

**Changes:**
- Updated database queries to include `name_mk` and `name_en` from menu_items
- Modified `getPromotionsByRestaurant()` method
- Modified `getPromotionById()` method
- Modified `getVisiblePromotionsByRestaurant()` method
- Menu items in promotions now display correctly in selected language

## Database Schema Changes

### New Tables

```sql
CREATE TABLE themes (
  id INT AUTO_INCREMENT PRIMARY KEY,
  name VARCHAR(100) NOT NULL,
  background_color VARCHAR(50) NOT NULL,
  text_color VARCHAR(50) NOT NULL,
  accent_color VARCHAR(50) NOT NULL,
  cta_color VARCHAR(50) NOT NULL,
  font_style VARCHAR(100) NOT NULL,
  is_default BOOLEAN DEFAULT FALSE,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);
```

### Modified Tables

```sql
-- Added to restaurants table
ALTER TABLE restaurants 
ADD COLUMN selected_theme INT NULL,
ADD CONSTRAINT fk_restaurants_theme FOREIGN KEY (selected_theme) REFERENCES themes(id) ON DELETE SET NULL;
```

### Seeded Data

Six default themes are automatically inserted on database initialization:
1. Default (id: 1) - Current orange theme
2. Classic & Elegant (id: 2)
3. Casual & Friendly (id: 3)
4. Modern & Minimal (id: 4)
5. Bold & Vibrant (id: 5)
6. Dark Mode (id: 6)

## API Endpoints

### Theme Endpoints

```
GET /api/themes
- Returns all available themes

GET /api/themes/:id
- Returns specific theme by ID

GET /api/restaurants/slug/:slug/theme
- Returns theme for specific restaurant
- Falls back to default theme if none selected
```

### Updated Endpoints

```
POST /api/admin/restaurants
- Now accepts selected_theme field

PUT /api/admin/restaurants/:id
- Now accepts selected_theme field
```

## Frontend Changes

### Admin Dashboard (`admin-dashboard.js`)

**New Features:**
- Theme selector dropdown in restaurant add/edit forms
- `loadThemesForForm()` method to populate theme options
- Theme selection saved when creating/updating restaurants

### Live Menu (`menu.js`)

**New Features:**
- `loadAndApplyTheme()` method loads restaurant theme
- `applyTheme()` method applies theme colors dynamically
- Helper methods `darkenColor()` and `lightenColor()` for color manipulation
- Dynamic style injection for theme-specific colors
- CSS custom properties for theme values

### Styles (`style.css`)

**New Features:**
- CSS custom properties for theme colors
- Root-level theme variable definitions
- Support for dynamic theme overrides

## Setup Instructions

### Database Setup

1. Run the database initialization:
```bash
node init-db.js
```

This will:
- Create all necessary tables including `themes`
- Add `selected_theme` column to restaurants
- Seed 6 default themes
- Create default admin user

### Running the Application

1. Install dependencies:
```bash
npm install
```

2. Configure environment variables in `.env`:
```
DB_HOST=your_database_host
DB_USER=your_database_user
DB_PASSWORD=your_database_password
DB_NAME=your_database_name
PORT=3000
```

3. Start with PM2:
```bash
pm2 start ecosystem.config.cjs
```

Or start in development:
```bash
npm start
```

## Usage

### For Super Admins

1. Log in to admin dashboard
2. Go to "Restaurants" section
3. Click "Add Restaurant" or edit existing restaurant
4. Select desired color theme from dropdown
5. Save changes

### For Restaurant Admins

1. Log in to admin dashboard
2. Edit your restaurant information
3. Select desired color theme from dropdown
4. Save changes

### For Customers

- Visit the live menu page
- Theme is automatically applied based on restaurant selection
- Colors, fonts, and styling adapt to selected theme
- Promotion items display in selected language (MK/EN)

## Technical Details

### Theme Application Flow

1. User loads menu page with restaurant slug
2. `loadAndApplyTheme()` fetches theme via API
3. Theme colors applied to CSS custom properties
4. Dynamic styles injected to override Tailwind classes
5. Body styles updated for immediate effect

### Localization in Promotions

1. Promotion items fetched with `name_mk` and `name_en`
2. `getLocalizedText()` method selects appropriate field
3. Falls back to default name if translation missing
4. Works consistently across all promotion displays

## Color Scheme Specifications

### 1. Default (Current)
- Background: `#ffffff` (White)
- Text: `#1f2937` (Gray-900)
- Accent: `#f97316` (Orange-600)
- CTA: `#f97316` (Orange-600)
- Font: `system-ui, -apple-system, sans-serif`

### 2. Classic & Elegant
- Background: `#faf9f6` (Ivory)
- Text: `#1e293b` (Charcoal)
- Accent: `#d4af37` (Gold)
- CTA: `#8b0000` (Dark Red)
- Font: `Georgia, serif`

### 3. Casual & Friendly
- Background: `#f5f5dc` (Beige)
- Text: `#3e2723` (Dark Brown)
- Accent: `#ff6347` (Tomato)
- CTA: `#228b22` (Forest Green)
- Font: `Verdana, sans-serif`

### 4. Modern & Minimal
- Background: `#ffffff` (Pure White)
- Text: `#000000` (Black)
- Accent: `#00bfa5` (Teal)
- CTA: `#0091ea` (Blue)
- Font: `Helvetica, Arial, sans-serif`

### 5. Bold & Vibrant
- Background: `#fffaf0` (Floral White)
- Text: `#1e3a8a` (Navy)
- Accent: `#dc2626` (Red)
- CTA: `#16a34a` (Green)
- Font: `Montserrat, sans-serif`

### 6. Dark Mode
- Background: `#1f2937` (Gray-800)
- Text: `#f3f4f6` (Gray-100)
- Accent: `#3b82f6` (Blue-500)
- CTA: `#ef4444` (Red-500)
- Font: `Inter, sans-serif`

## Files Modified

### Backend
- `src/database.js` - Added theme methods, updated promotion queries
- `src/server.js` - Added theme API endpoints

### Frontend
- `public/static/admin-dashboard.js` - Added theme selector to forms
- `public/static/menu.js` - Added theme loading and application
- `public/static/style.css` - Added theme CSS variables

### Configuration
- Database schema with themes table and restaurant theme column

## Future Enhancements

Potential improvements:
- Allow custom theme creation by super admins
- Theme preview before saving
- Per-category color customization
- Advanced typography options
- Theme templates for different restaurant types
- Import/export theme configurations

## Support

For issues or questions:
- Check server logs: `pm2 logs webapp --nostream`
- Verify database connection in `.env`
- Ensure themes are seeded: `SELECT * FROM themes;`
- Check browser console for frontend errors

## License

Proprietary - All rights reserved
